<?php

namespace App\Models;

use CodeIgniter\Model;
use Config\Database;

class ApiModel extends Model
{
    private $project_id;

    public function __construct($project_id)
    {
        $this->project_id = $project_id;
    }

    private function _sql_error($e)
    {
        if (API_DEBUG_LEVEL == 1) {
            $response = new \App\Libraries\ApiResponse();
            die($response->set_response_error(500, $e->getMessage()));
        } else {
            $response = new \App\Libraries\ApiResponse();
            die($response->set_response_error(500, 'Internal Server Error'));
        }
    }

    public function get_restaurant_details()
    {
        try {

            $db = Database::connect();
            $data = [];

            $params = [
                'project_id' => $this->project_id
            ];

            // get restaurant main information
            $results = $db->query("
                SELECT *
                FROM restaurants
                WHERE 1
                AND project_id = :project_id:
            ", $params);
            $data['restaurant_details'] = $results->getResult()[0];

            // get restaurant categories
            $results = $db->query("
                SELECT DISTINCT(p.category) category
                FROM products p JOIN restaurants r
                ON p.id_restaurant = r.id
                WHERE r.project_id = :project_id:
                AND p.deleted_at IS NULL
            ", $params);
            $data['products_categories'] = $results->getResult();

            // get restaurant products
            $results = $db->query("
                SELECT p.*
                FROM products p JOIN restaurants r
                ON p.id_restaurant = r.id
                WHERE r.project_id = :project_id:
                AND p.deleted_at IS NULL
            ", $params);
            $data['products'] = $results->getResult();

            return $data;
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {

            return $this->_sql_error($e);
        }
    }

    public function get_products_availability($data)
    {
        // get key->value pair of products (id->quantity)
        $products = [];
        foreach ($data as $product) {
            $id = $product['id_product'];
            $quantity = $product['quantity'];
            $products[$id] = $quantity;
        }

        // create a string with all the products ids
        $products_ids = implode(',', array_keys($products));

        // get products from database
        try {

            $db = Database::connect();
            $results = $db->query("
                SELECT * FROM products
                WHERE id IN ($products_ids)
                AND availability = 1
                AND stock > stock_min_limit
                AND deleted_at IS NULL
            ")->getResult();

            // check if the total products is equal to the total products in the request (order)
            if (count($results) != count($products)) {
                return [
                    'status' => 'error',
                    'message' => 'some products are not available'
                ];
            }

            // check if the quantity of each product is available
            foreach ($results as $product) {
                $quantity = $products[$id];
                if ($product->stock - $quantity <= $product->stock_min_limit) {
                    return [
                        'status' => 'error',
                        'message' => 'some products have not enough stock'
                    ];
                }
            }

            // all products are available
            return [
                'status' => 'success',
                'message' => 'all products are available'
            ];
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    public function get_last_order_number($id_restaurant)
    {
        try {

            $db = Database::connect();
            $params = [
                'id_restaurant' => $id_restaurant
            ];
            $results = $db->query("
                SELECT order_number
                FROM orders
                WHERE id_restaurant = :id_restaurant:
                ORDER BY order_number DESC
                LIMIT 1
            ", $params)->getResult();
            if (count($results) == 0) {
                return 0;
            } else {
                return $results[0]->order_number;
            }
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    public function add_order($id_restaurant, $machine_id, $total_price, $status, $order_number)
    {
        try {

            $db = Database::connect();
            $data = [
                'id_restaurant' => $id_restaurant,
                'machine_id' => $machine_id,
                'order_date' => date('Y-m-d H:i:s'),
                'order_status' => $status,
                'total_price' => $total_price,
                'order_number' => $order_number,
                'created_at' => date('Y-m-d H:i:s')
            ];
            $db->table('orders')->insert($data);
            return [
                'status' => 'success',
                'message' => 'order added successfully',
                'id' => $db->insertID(),
                'order_number' => $order_number
            ];
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    public function add_order_items($order_id, $order_items)
    {
        try {

            $db = Database::connect();
            $data = [];
            foreach ($order_items as $id_product => $item) {
                $tmp['id_order'] = $order_id;
                $tmp['id_product'] = $id_product;
                $tmp['price_per_unit'] = $item['price'];
                $tmp['quantity'] = $item['quantity'];
                $tmp['created_at'] = date('Y-m-d H:i:s');
                $data[] = $tmp;
            }
            $db->table('order_products')->insertBatch($data);
            return [
                'status' => 'success',
                'message' => 'order items added successfully'
            ];
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    public function get_pending_orders()
    {
        try {

            $db = Database::connect();
            $params = [
                'project_id' => $this->project_id
            ];
            $results = $db->query("
                SELECT
                    orders.*,
                    SUM(order_products.quantity) total_items
                FROM orders JOIN restaurants
                ON orders.id_restaurant = restaurants.id
                JOIN order_products
                ON orders.id = order_products.id_order
                WHERE orders.order_status = 'paid'
                AND restaurants.project_id = :project_id:
                AND orders.deleted_at IS NULL
                AND order_products.deleted_at IS NULL
                GROUP BY orders.id
            ", $params)->getResult();

            return [
                'status' => 'success',
                'message' => 'orders retrived successfully',
                'data' => $results
            ];
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    public function get_order_details($id)
    {
        try {

            $db = Database::connect();
            $params = [
                'id' => $id
            ];
            $results = $db->query("
            SELECT
                orders.*,
                SUM(order_products.quantity) total_items
            FROM orders JOIN order_products
            ON orders.id = order_products.id_order
            WHERE orders.id = :id:
            AND orders.deleted_at IS NULL
            ", $params)->getResult();
            return [
                'status' => 'success',
                'message' => 'order retrieved successfully',
                'data' => $results
            ];
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    public function delete_order($id)
    {
        try {

            $db = Database::connect();
            $params = [
                'id' => $id
            ];
            $results = $db->query("
                UPDATE orders
                SET deleted_at = NOW(),
                order_status = 'cancelled'
                WHERE id = :id:
            ", $params)->getResult();
            return [
                'status' => 'success',
                'message' => 'order deleted successfully',
                'data' => $results
            ];
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    public function get_order_details_with_products($id)
    {
        try {

            // first, get order general details
            $order_results = $this->get_order_details($id);
            if ($order_results['status'] == 'error') {
                return [
                    'status' => 'error',
                    'message' => 'error in order details'
                ];
            }

            $order['order_details'] = $order_results['data'][0];

            // second, get order products
            $db = Database::connect();
            $params = [
                'id' => $id
            ];
            $results = $db->query("
                SELECT
                    order_products.*,
                    products.name product_name,
                    products.category product_category
                FROM order_products JOIN products
                ON order_products.id_product = products.id
                WHERE order_products.id_order = :id:
                AND order_products.deleted_at IS NULL
            ", $params)->getResult();

            if (count($results) == 0) {
                return [
                    'status' => 'error',
                    'message' => 'error in order products'
                ];
            }

            $order['order_products'] = $results;

            // return order details with products
            return [
                'status' => 'success',
                'message' => 'order details with products retrieved successfully',
                'data' => $order
            ];
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    public function finish_order($id)
    {
        /* 
        atualizar o pedido para status "finished"
        atualizar o stock dos produtos que estão no pedido
        */
        try {

            $db = Database::connect();

            // -------------------------------------
            // 1. gets the order products id and quantity in the order
            $params = [
                'id' => $id
            ];
            $results = $db->query("
                SELECT
                    order_products.id_product,
                    order_products.quantity
                FROM order_products JOIN orders
                ON order_products.id_order = orders.id
                WHERE orders.id = :id:
                AND order_products.deleted_at IS NULL
            ", $params)->getResult();

            // 2. collect products ids and quantities in array
            $products = [];
            foreach ($results as $product) {
                $products[] = [
                    'id_product' => $product->id_product,
                    'quantity' => $product->quantity
                ];
            }

            // 3. check if the stock is enough for the order
            $products_availability = $this->get_products_availability($products);
            if ($products_availability['status'] == 'error') {
                return $products_availability;
            }

            // 4. build the sql query with multiple updates
            $ids = [];
            foreach ($results as $product) {
                $ids[] = $product->id_product;
            }
            $sql = "UPDATE products SET stock = CASE id ";
            foreach ($products as $product) {
                $sql .= "WHEN {$product['id_product']} THEN stock - {$product['quantity']} ";
            }
            $sql .= "END WHERE id IN(" . implode(',', $ids) . ")";

            // 5. execute the query
            $db->query($sql);

            // -------------------------------------
            // 6. updates order status to 'finished'
            $params = [
                'id' => $id
            ];
            $db->query("
                UPDATE orders
                SET order_status = 'finished',
                updated_at = NOW()
                WHERE id = :id:
            ", $params);

            return [
                'status' => 'success',
                'message' => 'order finished successfully',
                'data' => []
            ];

        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }
}
